module net.BurtonRadons.spyl.parser;

private import net.BurtonRadons.spyl.lexer;

class Parser : Lexer
{
    private import net.BurtonRadons.spyl.expression;
    private import net.BurtonRadons.spyl.statement;
    private import net.BurtonRadons.spyl.value;
    private import net.BurtonRadons.spyl.mark;
    
    this (char [] sourceName, char [] content)
    {
        super (sourceName, content);
    }
    
    BlockStatement parseModule ()
    {
        BlockStatement statement;
        
        statement = new BlockStatement (mark);
        
        while (1)
        {
            if (peekToken ().type == TokenType.EOF)
                return statement;
            
            Statement add;
            
            statement.add (add = parseStatement ());
            if (add === null)
                return null;
        }
    }
    
    BlockStatement parseBlockStatement ()
    {
        BlockStatement statement;
        
        Token token = peekToken ();
        expectToken (TokenType.CurlyBraceLeft);
        statement = new BlockStatement (token.mark);
        
        while (1)
        {
            Token token = peekToken ();
            
            if (token.type == TokenType.CurlyBraceRight || token.type == TokenType.EOF)
            {
                expectToken (TokenType.CurlyBraceRight);
                return statement;
            }
            
            statement.add (parseStatement ());
        }
        
        return statement;
    }
    
    /** Parse a statement. */
    Statement parseStatement ()
    {
        Token token = peekToken ();
        
        if (token.type == TokenType.EOF)
        {
            error ("Unexpected end-of-file");
            return null;
        }
        
        if (token.type == TokenType.CurlyBraceLeft)
            return parseBlockStatement ();
        
        if (token.type == TokenType.Id)
        {
            Expression expression = parseExpression ();
            
            if (!expression)
                return null;
            if (!expectToken (TokenType.SemiColon))
                return null;
            return new ExpressionStatement (expression.mark, expression);
        }
        
        error ("Expected statement but got " ~ Token.typeName (token.type) ~ ".");
        
        return null;
    }
    
    Expression parseExpression ()
    {
        return parseAssignExpression ();
    }
    
    Expression parseAssignExpression ()
    {
        Expression result, other;
        
        result = parseAddExpression ();
        if (!result)
            return null;
        
        while (1)
        {
            Mark mark = peekToken ().mark;
            
            if (foundToken (TokenType.Assign))
            {
                if ((other = parseAssignExpression ()) === null)
                    return null;
                result = new AssignExpression (mark, result, other);
            }
            else if (foundToken (TokenType.TildeAssign))
            {
                if ((other = parseAssignExpression ()) === null)
                    return null;
                result = new CatenateAssignExpression (mark, result, other);
            }
            else
                return result;
        }
    }
    
    Expression parseAddExpression ()
    {
        Expression result, other;
        
        if ((result = parseMultiplyExpression ()) === null)
            return null;
        
        while (1)
        {
            Mark mark = peekToken ().mark;
            
            if (foundToken (TokenType.Add))
            {
                if ((other = parseMultiplyExpression ()) === null)
                    return null;
                result = new AddExpression (mark, result, other);
            }
            else if (foundToken (TokenType.Minus))
            {
                if ((other = parseMultiplyExpression ()) === null)
                    return null;
                result = new SubtractExpression (mark, result, other);
            }
            else if (foundToken (TokenType.Tilde))
            {
                if ((other = parseMultiplyExpression ()) === null)
                    return null;
                result = new CatenateExpression (mark, result, other);
            }
            else
                return result;
        }
    }
    
    Expression parseMultiplyExpression ()
    {
        Expression result, other;
        
        if ((result = parseExponentExpression ()) === null)
            return null;
        
        while (1)
        {
            Mark mark = peekToken ().mark;
            
            if (foundToken (TokenType.Asterisk))
            {
                if ((other = parseExponentExpression ()) === null)
                    return null;
                result = new MultiplyExpression (mark, result, other);
            }
            else
                return result;
        }
    }
    
    Expression parseExponentExpression ()
    {
        Expression result, other;
        
        if ((result = parseUnaryExpression ()) === null)
            return null;
        
        while (1)
        {
            Mark mark = peekToken ().mark;
            
            if (foundToken (TokenType.AsteriskAsterisk))
            {
                if ((other = parseUnaryExpression ()) === null)
                    return null;
                result = new ExponentExpression (mark, result, other);
            }
            else
                return result;
        }
    }
    
    Expression parseUnaryExpression ()
    {
        Mark mark = peekToken ().mark;
        Expression result;
        
        if (foundToken (TokenType.Minus))
        {
            result = parseUnaryExpression ();
            if (result)
                result = new NegateExpression (mark, result);
            return result;
        }
        
        return parsePostfixExpression ();
    }
    
    Expression parsePostfixExpression ()
    {
        Expression result;
        
        result = parsePrimaryExpression ();
        if (!result)
            return null;
        
        while (1)
        {
            Mark mark = peekToken ().mark;
            
            if (foundToken (TokenType.ParenthesisLeft))
            {
                Expression [] arguments;
                Expression add;
                
                while (1)
                {
                    if (foundToken (TokenType.ParenthesisRight))
                        break;
                    arguments ~= add = parseAssignExpression ();
                    if (add === null)
                        return null;
                    if (foundToken (TokenType.ParenthesisRight))
                        break;
                    if (!expectToken (TokenType.Comma))
                        return null;
                }
                
                result = new CallExpression (mark, result, arguments, null);
            }
            else if (foundToken (TokenType.Dot))
            {
                char [] text = peekToken ().text;
                
                if (!expectToken (TokenType.Id))
                    return null;
                result = new AttrExpression (mark, result, text);
            }
            else if (foundToken (TokenType.BracketLeft))
            {
                Expression index;
                Expression min = null, max = null;
                
                if (foundToken (TokenType.DotDot))
                {
                dotDot:
                    if (peekToken ().type != TokenType.BracketRight)
                        max = parseAssignExpression ();
                    else
                        max = new ValueExpression (mark, Null);
                        
                    if (!max || !expectToken (TokenType.BracketRight))
                        return null;
                    if (!min)
                        min = new ValueExpression (mark, Null);
                    result = new SliceExpression (mark, result, min, max);
                    continue;
                }
                
                min = index = parseAssignExpression ();
                if (!index)
                    return null;
                
                if (foundToken (TokenType.DotDot))
                    goto dotDot;
                else
                {
                    if (!expectToken (TokenType.BracketRight))
                        return null;
                    result = new IndexExpression (mark, result, index);
                }
            }
            else
                return result;
        }
    }
    
    Expression parsePrimaryExpression ()
    {
        Expression result;
        Token token = peekToken ();
        Mark mark = token.mark;
        
        if (foundToken (TokenType.ParenthesisLeft))
        {
            result = parseExpression ();
            if (result === null || !expectToken (TokenType.ParenthesisRight))
                return null;
            return result;
        }
        
        if (foundToken (TokenType.BracketLeft))
        {
            Expression [] values;
            
            while (1)
            {
                if (foundToken (TokenType.BracketRight))
                    break;
                result = parseAssignExpression ();
                if (result === null)
                    return null;
                values ~= result;
                if (foundToken (TokenType.BracketRight))
                    break;
                if (!expectToken (TokenType.Comma))
                    return null;
            }
            
            return new ArrayExpression (mark, values);
        }
        
        if (token.type == TokenType.Id)
            result = new SymbolExpression (token.mark, token.text);
        else if (token.type == TokenType.String)
            result = new ValueExpression (token.mark, new StringValue (token.text.dup));
        else if (token.type == TokenType.Integer)
            result = new ValueExpression (token.mark, new IntegerValue (token.integerValue));
        else if (token.type == TokenType.Float)
            result = new ValueExpression (token.mark, new FloatValue (token.floatValue));
        else
        {
            error ("Do not know how to handle " ~ Token.typeName (token.type) ~ " as a unary expression.");
            return null;
        }
        
        nextToken ();
        return result;
    }
}
